/*
 * Copyright (c) 2015 Adobe Systems.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, regexp: true, indent: 4, maxerr: 50 */
/*global require, exports, process, Promise:true */
// "strict": [2, "global"]
"use strict";

var USBMUXConnection  = require("./USBMUXConnection").USBMUXConnection,
    USBMUXSocket      = require("./USBMUXSocket"),
    CONSTANTS         = require("./USBMUXConstants"),
    USBMUXHelper      = require("./USBMUXHelper"),
    Logger            = require("./Logging"),
    Promise           = require("bluebird");

var logger = Logger.createLogger({name: "USBMUXClient", level: "debug"});

/**
 * Create a new USBMUXSocket.
 * @param socketOptions provide information about the unix socket or port to connect to usbmuxd.
 * This is /var/run/usbmuxd on OSX and port 27015 on Windows.
 * @constructor
 */
function USBMUXClient(socketOptions) {
    logger.debug("created");

    this.socketOptions = socketOptions || USBMUXHelper.getUSBMUXSocketOptions(process.platform);
}

/**
 * Return the attached {USBMUXDevice} devices.
 */
Object.defineProperty(USBMUXClient.prototype, "devices", {
    get: function () { logger.debug("get devices"); return this.connection.devices; }
});

/**
 * Create a new "control" socket to listen for device attach and remove events
 *
 * @param @callback {Function=} cb callback to handle the DEVICE_ADDED and DEVICE_REMOVED events.
 * @returns {Promise} resolves when the connection to the usbmuxd service has been established.
 */
USBMUXClient.prototype.watch = function (cb) {
    logger.debug("watch");

    var eventHandler = cb || function() { };

    return new Promise(function (resolve, reject) {
        var socket = USBMUXSocket.create(this.socketOptions, "control", new USBMUXSocket.MessageBuffer());
        this.connection = new USBMUXConnection(socket, CONSTANTS.USBMUX_PROTOCOL_VERSION_PLIST);

        this.connection.listen(eventHandler).then(function() {
            resolve();
        }).catch(function (err) {
            reject(err);
        });
    }.bind(this));
};

/**
 * Connect to a port on the specified device. The app needs to run on the device and needs to listen to the port.
 *
 * @param {!USBUXDevice} device the device that is running the app
 * @param {!number} port    the port that the app listens on the device.
 *
 * @returns {Promise}   resolve with a socket to the app on the device, reject with err
 */
USBMUXClient.prototype.connectToDevice = function (device, port) {
    return new Promise(function (resolve, reject) {
        var socket = USBMUXSocket.create(this.socketOptions, "device", new USBMUXSocket.MessageBuffer());
        var connection = new USBMUXConnection(socket, CONSTANTS.USBMUX_PROTOCOL_VERSION_PLIST);

        connection.connectToDevice(device, port).then(function (_socket) {
            resolve(_socket);
        }).catch(function (error) {
            reject(error);
        });
    }.bind(this));
};

/**
 * @constant {exports.USBMUX_PROTOCOL_VERSION_BINARY}   the version of the protocol
 */
USBMUXClient.prototype.version = CONSTANTS.USBMUX_PROTOCOL_VERSION_BINARY;

/**
 * Create an instance of the USBMUX client to establish a communication with the usbmuxd service.
 *
 * @see {@link USBMUXClient}
 * @returns {USBMUXClient} a new instance of a USBMUXClient
 */
function create(socketOptions) {
    return new USBMUXClient(socketOptions);
}

// API
exports.create = create;
